#include<stdio.h>
#include<sys/times.h>

#include<mpi.h>

#include<X11/Xlib.h>
#include<X11/Xutil.h>
#include<X11/Xos.h>

#define WIN_SIZE_W 600
#define WIN_SIZE_H 600

enum TypeMsg {
  MSG_DEMANDE,
  MSG_TRAITEMENT,
  MSG_RESULTAT,
  MSG_AUCUN_TRAITEMENT
};

typedef struct {
  double real, imag;
} complex ;

static int init_display(const uint width, const uint height,
			Display ** display, Window* win, GC* gc, 
			unsigned long* min_color, unsigned long* max_color);

static int calcPixel(complex c, int maxiter);

int main(int argc, char** argv){
  
  int size, rank;
  MPI_Status status;

  int root = 0;
  
  unsigned int width = WIN_SIZE_W;
  unsigned int height = WIN_SIZE_H;
  double x, y, z_size;
  int maxiter = 100;

  // Range to examine
  double real_min = -2, real_max = 2;
  double imag_min = -2, imag_max = 2;

  char *buffer;
  int granularite = 100;
  int size_buf;

  int type_msg, position;

  // Initialize MPI
  MPI_Init(&argc, &argv);
  
  // Recuperation du rang du nombre processus
  MPI_Comm_size(MPI_COMM_WORLD, &size);
  // Recuperation du rang du processus
  MPI_Comm_rank(MPI_COMM_WORLD, &rank);
  
  if((argc > 3 && argc < 6) || (argc > 1 && strcmp(argv[1],"-h")==0)){
    if(rank == root){
      fprintf(stdout, "USAGE %s : [granularite]  [maxiter] [x y z_size]\n", argv[0]);
      fprintf(stdout, "           granularite : nombre de case a calculer par traitement\n");
      fprintf(stdout, "           maxiter : nombre maximum d'iteration\n");
      fprintf(stdout, "           x,y : point central d'agrandissement du rendu\n");
      fprintf(stdout, "           z_size : taille de la zone a afficher\n");
    }
    MPI_Finalize();
    return -1;
  }

  // Recuperation des parametres
  if(argc>1){
    granularite = atoi(argv[1]);    
  }
  size_buf = (4+granularite)*sizeof(int);
  buffer = (char *)malloc(size_buf);
  if(argc > 2){
    maxiter = atoi(argv[2]);
  }
  if(argc > 3){
    x = atof(argv[3]);
    y = atof(argv[4]);
    z_size = atof(argv[5]);
    real_min = x - z_size/2;
    real_max = real_min + z_size;
    imag_min = y - z_size/2;
    imag_max = imag_min + z_size;
  }

  // Processus fils calcul les donnees a envoyer au maitre
  if (rank != root){

    // Calculate and draw points
    complex c;
    
    // Compute factors to scale computational region to window
    double scale_real = (double) (real_max - real_min) / (double) width;
    double scale_imag = (double) (imag_max - imag_min) / (double) height;

    int encore=1;

    int i,j, res;

    // Tant qu'il y a des traitements a faire
    while(encore){
      // On demande un traitement au maitre
      position=0;
      type_msg = MSG_DEMANDE;
      MPI_Pack(&type_msg, 1, MPI_INT, buffer, size_buf, &position, MPI_COMM_WORLD);
      MPI_Pack(&rank, 1, MPI_INT, buffer, size_buf, &position, MPI_COMM_WORLD);
      MPI_Send(buffer, position, MPI_PACKED, root, 1, MPI_COMM_WORLD);

      // On recoit sa reponse
      position=0;
      MPI_Recv(buffer, size_buf, MPI_PACKED, root, 1, MPI_COMM_WORLD, &status);
      MPI_Unpack(buffer, size_buf, &position, &type_msg, 1, MPI_INT, MPI_COMM_WORLD);

      int min_case, max_case;

      switch(type_msg){
      case MSG_TRAITEMENT:
	
	// Recupere les donnees
	MPI_Unpack(buffer, size_buf, &position, &min_case, 1, MPI_INT, MPI_COMM_WORLD);
	MPI_Unpack(buffer, size_buf, &position, &max_case, 1, MPI_INT, MPI_COMM_WORLD);

	// Message a envoye au maitre
	position=0;
	type_msg = MSG_RESULTAT;
	MPI_Pack(&type_msg, 1, MPI_INT, buffer, size_buf, &position, MPI_COMM_WORLD);
	MPI_Pack(&rank, 1, MPI_INT, buffer, size_buf, &position, MPI_COMM_WORLD);
	MPI_Pack(&min_case, 1, MPI_INT, buffer, size_buf, &position, MPI_COMM_WORLD);
	MPI_Pack(&max_case, 1, MPI_INT, buffer, size_buf, &position, MPI_COMM_WORLD);

	int k;
	for(k=min_case;k<=max_case;k++){
	  i = k/height;
	  j = k%height;
	  // Effectue le traitement
	  c.real = real_min + ((double) j * scale_real);
	  c.imag = imag_min + ((double) (height-1-i) * scale_imag);
	  res = calcPixel(c, maxiter);
	  MPI_Pack(&res, 1, MPI_INT, buffer, size_buf, &position, MPI_COMM_WORLD);
	}

	// Envoi du resultat au maitre
	MPI_Send(buffer, position, MPI_PACKED, root, 1, MPI_COMM_WORLD);	
	break;
      case MSG_AUCUN_TRAITEMENT:
	//fprintf(stdout, "Fin des traitements pour rank=%d\n", rank);
	encore=0;
	break;
      }
    }


  }else{
    double t_debut = MPI_Wtime();

    Display *display;
    Window win;
    GC gc;

    unsigned long min_color, max_color;
    
    int ret;
    ret = init_display(width, height, &display, &win, &gc, &min_color, &max_color);
    //fprintf(stdout, "Initialisation de l'affichage : %d\n", ret);
    if (ret < 0){
      fprintf(stderr, "Erreur  l'nitialisation de l'affichage\n");
      MPI_Finalize();
      return -1;
    }

    // Compute factor for color scaling
    double scale_color = (double) (max_color - min_color) / 
      (double) (maxiter - 1);
    long color;

    int e_rank;
    
    int nb_worker=size-1;
    int num_case=0, min_case, max_case;
    int i, j, res;

    while(nb_worker){
      // On attend un message d'un worker
      position=0;
      MPI_Recv(buffer, size_buf, MPI_PACKED, MPI_ANY_SOURCE, 1, MPI_COMM_WORLD, &status);
      MPI_Unpack(buffer, size_buf, &position, &type_msg, 1, MPI_INT, MPI_COMM_WORLD);
      MPI_Unpack(buffer, size_buf, &position, &e_rank, 1, MPI_INT, MPI_COMM_WORLD);

      switch(type_msg){
      case MSG_DEMANDE:
	position=0;
	// Test si il y a encore des calculs a realiser
	if(num_case <= width*height){
	  // Envoi des informations du calcul
	  type_msg=MSG_TRAITEMENT;
	  position=0;
	  min_case = num_case;
	  max_case = num_case + granularite - 1;
	  if(max_case > width*height){
	    max_case = width*height;
	  }
	  
	  MPI_Pack(&type_msg, 1, MPI_INT, buffer, size_buf, &position, MPI_COMM_WORLD);
	  MPI_Pack(&min_case, 1, MPI_INT, buffer, size_buf, &position, MPI_COMM_WORLD);
	  MPI_Pack(&max_case, 1, MPI_INT, buffer, size_buf, &position, MPI_COMM_WORLD);
	  num_case = max_case + 1;
	}else{
	  type_msg=MSG_AUCUN_TRAITEMENT;
	  position=0;
	  MPI_Pack(&type_msg, 1, MPI_INT, buffer, size_buf, &position, MPI_COMM_WORLD);
	  nb_worker--;
	}
	MPI_Send(buffer, position, MPI_PACKED, e_rank, 1, MPI_COMM_WORLD);
	break;
      case MSG_RESULTAT:
	// Reception d'un resultat de traitement
	MPI_Unpack(buffer, size_buf, &position, &min_case, 1, MPI_INT, MPI_COMM_WORLD);
	MPI_Unpack(buffer, size_buf, &position, &max_case, 1, MPI_INT, MPI_COMM_WORLD);
	
	int k;
	// Lecture et affichage des resultats
	for(k=min_case;k<=max_case;k++){
	  MPI_Unpack(buffer, size_buf, &position, &res, 1, MPI_INT, MPI_COMM_WORLD);

	  color = ((long)(res-1) * scale_color) + min_color;
	  
	  i = k/height;
	  j = k%height;
	  XSetForeground (display, gc, color);
	  XDrawPoint (display, win, gc, j, i);
	  XFlush (display);
	}
      }
    }

    double t_fin = MPI_Wtime();
    fprintf(stdout, "ChargeDynamique;%d;%d;%d;%f\n", size, granularite, maxiter, t_fin - t_debut);
    
    // Processus maitre effectue l'affichage du resultat
    /*
      fprintf(stdout, "Creation d'une image de %dX%d pixels\n", width, height);
      fprintf(stdout, "Center=(%f,%f), size=%d\n", (real_max + real_min)/2,
      (imag_max + imag_min)/2, (int)(real_max - real_min));
      fprintf(stdout, "Nombre maximum d'iterations = %d\n", maxiter);
      fprintf(stdout, "Granularite = %d\n", granularite);
      fprintf(stdout, "Temps d'execution en secondes = %f sec\n", t_fin - t_debut);
      fprintf(stdout, "Affichage de l'image\n");
      for(i=10;i>0;i--){
      fprintf(stdout, "%d...\n", i);
      sleep(1);
      }
      fprintf(stdout, "Fin du traitement\n");
    */

  }

  MPI_Finalize();
  return(0);
}

int init_display(const unsigned int width, const unsigned int height,
		 Display ** display, Window* win, GC* gc, 
		 unsigned long* min_color, unsigned long* max_color) {
  
  // Variables for graphical display
  
  unsigned int
    x = 0, y = 0,			// window position
    border_width = 4,			// border width in pixels
    disp_width, disp_height,		// size of screen
    screen;				// which screen
  
  char *window_name = "Mandelbrot Set",
    *disp_name = NULL;
  unsigned long valuemask = 0;
  XGCValues values;
  
  unsigned long white, black;	// white, black pixel values
 
  // Connect to Xserver
  if ((*display = XOpenDisplay(disp_name)) == NULL){
    fprintf (stderr, "drawon: cannot connect to X server %s\n",
	     XDisplayName (disp_name) );
    
    return -1;
  }
        
  /// Initialize for graphical display 
  screen = DefaultScreen (*display);
  disp_width = DisplayWidth (*display, screen);
  disp_height = DisplayHeight (*display, screen);

  *win = XCreateSimpleWindow (*display, RootWindow (*display, screen),
			     x, y, width, height, border_width, 
			     BlackPixel (*display, screen), 
			     WhitePixel (*display, screen));

  XStoreName(*display, *win, window_name);

  *gc = XCreateGC (*display, *win, valuemask, &values);
				// graphics context
  white = WhitePixel (*display, screen);	// color value for white
  black = BlackPixel (*display, screen);	// color value for black

  XSetBackground (*display, *gc, white);
  XSetForeground (*display, *gc, black);

  XMapWindow (*display, *win);
  XSync(*display, 0);
  
  // Get min and max for range of color values -- assumed to
  //   be defined by "white", "black"
  *min_color = (white > black) ? black : white;
  *max_color = (white > black) ? white : black;

  // Pause -- a crude way to reduce the chance that we'll miss
  //   the first few rows of output.
  usleep(200000);  	

  return EXIT_SUCCESS;
}

// number of iteration for one point computation
int calcPixel(complex c, int maxiter){
     // Calculate z0, z1, .... until divergence or maximum iterations
  complex z;
  float lengthsq, temp;
  int count = 0;
  z.real = z.imag = 0;
  
  do  {
    temp = z.real*z.real - z.imag*z.imag + c.real;
    z.imag = 2.0*z.real*z.imag + c.imag;
    z.real = temp;
    lengthsq = z.real*z.real + z.imag*z.imag;
    count++;
  } while (lengthsq < 4.0 && count < maxiter);
  return count;
}
