//
// Sequential Mandelbrot program 
// 
// This program computes and displays all or part of the Mandelbrot 
// set.  By default, it examines all points in the complex plane
// that have both real and imaginary parts between -2 and 2.  
// Command-line parameters allow zooming in on a specific part of
// this range.
// 
// Usage:
//   mandelbrot [x y size] [maxiter]
// where 
//   x, y, and size specify the range to examine (a square of size
//     "size", centered at x + iy -- by default, a square of size 4
//     centered at the origin)
//   maxiter denotes the maximum number of iterations at each point
//     (default 100)
// Arguments are optional.
// 
// Input:  none, except the optional command-line arguments
// Output: a graphical display as described in Wilkinson & Allen,
//   displayed using the X Window system, plus text output to
//   standard output showing the above parameters, plus execution
//   time in seconds.
// 
// 
// Original code obtained from Web site for Wilkinson and Allen's
// text on parallel programming:
// http://www.cs.uncc.edu/~abw/parallel/par_prog/
// 
// Reformatted and revised by B. Massingill, Jan 2001 and Feb 2002
//

#include <stdio.h>
#include <string.h>
#include <math.h>
#include <stdlib.h>
#include <sys/times.h>
#include <time.h>
#include <unistd.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>

// Structure definition for complex numbers
// (We could use a class, but this is noticeably faster.)
typedef struct {
  double real, imag;
} complex ;

// Shorthand for some commonly-used types
typedef unsigned int uint;
typedef unsigned long ulong;

// ---- Function declarations ------------------------------------------

int setup(const uint width, const uint height,
	  Display ** display, Window* win, GC* gc, 
	  ulong* min_color, ulong* max_color);
void interact(Display* display, Window win, 
	      const uint width, const uint height,
	      const double real_min, const double real_max,
	      const double imag_min, const double imag_max);

// number of iteration for one point computation

int calcPixel(complex c, int maxiter){
     // Calculate z0, z1, .... until divergence or maximum iterations
  complex z;
  float lengthsq, temp;
  int count = 0;
  z.real = z.imag = 0;
  
  do  {
    temp = z.real*z.real - z.imag*z.imag + c.real;
    z.imag = 2.0*z.real*z.imag + c.imag;
    z.real = temp;
    lengthsq = z.real*z.real + z.imag*z.imag;
    count++;
  } while (lengthsq < 4.0 && count < maxiter);
  return count;
}


// ---- Main program ---------------------------------------------------
int main (int argc, char *argv[]) {

  const uint width = 600;	// dimensions of display window,
  const uint height = 600;	//   in pixels

  // Check command-line arguments, if any
  // Usage is:
  //   mandelbrot [x y size] [maxiter]
  //   or
  //   mandelbrot -h to show usage message

  if (((argc > 1) && (strcmp(argv[1], "-h") == 0))
      || ((argc > 2) && (argc < 5))) {
    printf("Usage:  mandelbrot [maxiter] [x y size]\n");
    exit (EXIT_FAILURE);
  }
  uint i,j;
  // Set default values
  int maxiter = 100;	
  double			// range to examine
    real_min = -2, real_max = 2,
    imag_min = -2, imag_max = 2;

  // Process command-line arguments, if any
  if (argc > 2) {
    // argc >= 4 -- we checked this earlier
    double x = atof(argv[2]);
    double y = atof(argv[3]);
    double size = atof(argv[4]);
    real_min = x - size/2;
    real_max = real_min + size;
    imag_min = y - size/2;
    imag_max = imag_min + size;
  }
  if (argc > 1) 
    maxiter = atoi(argv[1]);

 // Initialize for graphical display

  Display *display;
  Window win;
  GC gc;
  ulong min_color, max_color;

  if (setup(width, height, &display, &win, &gc, &min_color, &max_color) !=
      EXIT_SUCCESS)
    exit(EXIT_FAILURE);
  
  // Calculate and draw points
  complex c;

  // Compute factors to scale computational region to window
  double scale_real = (double) (real_max - real_min) / (double) width;
  double scale_imag = (double) (imag_max - imag_min) / (double) height;

  // Compute factor for color scaling
  double scale_color = (double) (max_color - min_color) / 
		       (double) (maxiter - 1);
 
  for (j = 0; j < height; ++j) 
    for (i = 0; i < width; ++i) {
      // Scale display coordinates to actual region 
      c.real = real_min + ((double) i * scale_real);
      c.imag = imag_min + ((double) (height-1-j) * scale_imag);
				// height-1-j so y axis displays
				// with larger values at top
      //printf("Complex pix=%d, real=%f, imag=%f\n", j*width+i, c.real, c.imag);

      int k = calcPixel(c, maxiter);
      // Scale color and display point 
      long color = ((long)(k-1) * scale_color) + min_color;
      
      XSetForeground (display, gc, color);
      XDrawPoint (display, win, gc, i, j);
    }

  // Be sure all output is written

  XFlush (display);

  // Calculate execution time for program and produce text output
  
  struct tms t;
  times(&t);
  /*printf("center = ( %f  , %f ), size= %d\n", (real_max + real_min)/2,
	 (imag_max + imag_min)/2, real_max - real_min);
	 printf("maximum iterations = %d\n", maxiter);*/
  double time = 
    ((double) (t.tms_utime + t.tms_stime + t.tms_cutime + t.tms_cstime))/100;
  //printf("execution time in seconds = %f \n", time);
  
  fprintf(stdout, "Sequentiel;1;%d;%d;%f\n", height*width, maxiter, time);

  // Wait, then exit program 
  // sleep(10);
  return EXIT_SUCCESS;
}

// ---- Function definitions -------------------------------------------

// initialize for graphical display:
// pre:  width, height are dimensions of display, in pixels
// post:  remaining parameters have been set to values needed by
//          caller
//        return value is EXIT_SUCCESS if all goes well,
//          EXIT_FAILURE otherwise 
int setup(const uint width, const uint height,
	  Display ** display, Window* win, GC* gc, 
	  ulong* min_color, ulong* max_color) {

  // Variables for graphical display

  uint
    x = 0, y = 0,			// window position
    border_width = 4,			// border width in pixels
    disp_width, disp_height,		// size of screen
    screen;				// which screen

  char *window_name = "Mandelbrot Set",
    *disp_name = NULL;
  ulong valuemask = 0;
  XGCValues values;

  ulong white, black;	// white, black pixel values
 
  // Connect to Xserver

  if ( (*display = XOpenDisplay (disp_name)) == NULL ) {
     fprintf (stderr, "drawon: cannot connect to X server %s\n",
				XDisplayName (disp_name) );
	
    return EXIT_FAILURE;
  }
        
  /// Initialize for graphical display 

  screen = DefaultScreen (*display);
  disp_width = DisplayWidth (*display, screen);
  disp_height = DisplayHeight (*display, screen);

  *win = XCreateSimpleWindow (*display, RootWindow (*display, screen),
			     x, y, width, height, border_width, 
			     BlackPixel (*display, screen), 
			     WhitePixel (*display, screen));

  XStoreName(*display, *win, window_name);

  *gc = XCreateGC (*display, *win, valuemask, &values);
				// graphics context
  white = WhitePixel (*display, screen);	// color value for white
  black = BlackPixel (*display, screen);	// color value for black

  XSetBackground (*display, *gc, white);
  XSetForeground (*display, *gc, black);

  XMapWindow (*display, *win);
  XSync(*display, 0);
  
  // Get min and max for range of color values -- assumed to
  //   be defined by "white", "black"
  *min_color = (white > black) ? black : white;
  *max_color = (white > black) ? white : black;

  // Pause -- a crude way to reduce the chance that we'll miss
  //   the first few rows of output.
  usleep(200000);  	

  return EXIT_SUCCESS;
}

